package org.msh.tb.bd.tbforms.dhis2.tb10.blockbuilders;

import org.msh.tb.bd.tbforms.dhis2.TBFormBlockBuilder;
import org.msh.tb.bd.tbforms.dhis2.data.DataValuePayload;
import org.msh.tb.bd.tbforms.query.TbFormQuery;
import org.msh.tb.bd.tbforms.query.tb10.TBForm10Block1Query;
import org.msh.tb.entities.enums.CaseDefinition;
import org.msh.tb.entities.enums.Gender;
import org.msh.tb.entities.enums.InfectionSite;
import org.msh.tb.entities.enums.PatientType;

/**
 * Created by Mauricio on 17/02/2017.
 * Query database and mount TB Form 10 Block 1 Payload to be sent to DHIS2 APIs
 */
public class TBForm10Block1Builder extends TBFormBlockBuilder {

    private TBForm10Block1Query formQuery;

    @Override
    protected void loadAllFieldsAsZero(DataValuePayload payload) {
        for (DataElementTB10Block1 de : DataElementTB10Block1.values()) {
            for (COCAgeGenderBlock1 coc : COCAgeGenderBlock1.values()) {
                payload.mergeDataValue(de.dhis2Id, coc.dhis2Id, new Long(0));
            }
        }
    }

    @Override
    protected TbFormQuery getBlockQuery() {
        if (formQuery == null) {
            formQuery = new TBForm10Block1Query();
        }

        return formQuery;
    }

    @Override
    protected void mountResultPayload(Object[] result, DataValuePayload payload) {
        PatientType patientType = (PatientType) result[0];
        PatientType previouslyTreatedType = (PatientType) result[1];
        CaseDefinition caseDefinition = (CaseDefinition) result[2];
        InfectionSite infectionSite = (InfectionSite) result[3];
        Integer age = (Integer) result[4];
        Gender gender = (Gender) result[5];
        Long count = (Long) result[6];

        // find the DHIS2 data element and category id
        String dataElementId = DataElementTB10Block1.findId(infectionSite, caseDefinition, patientType, previouslyTreatedType);
        String categoryOptionComboId = COCAgeGenderBlock1.findId(age, gender);

        // merge payload result
        payload.mergeDataValue(dataElementId, categoryOptionComboId, count);
    }

    /**
     * Catalog that holds the ids of DHIS2 data element used by TB 10 Form Block 1
     */
    public enum DataElementTB10Block1 {

        // Pulmonary Bacteriologically Confirmed
        PULMONARY_BACTER_TREATHISTUNKNOWN("nuigb7nrgyy", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.UNKNOWN_PREVIOUS_TB_TREAT, null),
        PULMONARY_BACTER_NEW("ZisIEZC34Z6", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.NEW, null),
        PULMONARY_BACTER_RELAPSE("dC6JgYep2S6", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.PREVIOUSLY_TREATED, PatientType.RELAPSE),
        PULMONARY_BACTER_TREATAFTERFAILURE("uEWjLnNUiXX", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_FAILURE),
        PULMONARY_BACTER_TREATAFTERLOSSF("Ng8yglkNz5T", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_LOSS_FOLLOW_UP),
        PULMONARY_BACTER_OTHER("UwSI3k5euNE", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED, PatientType.PREVIOUSLY_TREATED, PatientType.OTHER_PREVIOUSLY_TREATED),

        // Pulmonary Clinically Diagnosed
        PULMONARY_CLIDIAG_TREATHISTUNKNOWN("dSaC3K0n0Qm", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.UNKNOWN_PREVIOUS_TB_TREAT, null),
        PULMONARY_CLIDIAG_NEW("zUszTiN1dC1", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.NEW, null),
        PULMONARY_CLIDIAG_RELAPSE("i9HYXjfVVcY", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.PREVIOUSLY_TREATED, PatientType.RELAPSE),
        PULMONARY_CLIDIAG_TREATAFTERFAILURE("UJUwT06Z3zp", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_FAILURE),
        PULMONARY_CLIDIAG_TREATAFTERLOSSF("fnwOKFyd2T6", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_LOSS_FOLLOW_UP),
        PULMONARY_CLIDIAG_OTHER("LO03cqjku3f", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED, PatientType.PREVIOUSLY_TREATED, PatientType.OTHER_PREVIOUSLY_TREATED),

        // Extra-Pulmonary Bacteriologically Confirmed or Clinically Diagnosed
        EXTRAPUL_TREATHISTUNKNOWN("GUUbqhj5ozd", InfectionSite.EXTRAPULMONARY, null, PatientType.UNKNOWN_PREVIOUS_TB_TREAT, null),
        EXTRAPUL_NEW("obu6z2HObRM", InfectionSite.EXTRAPULMONARY, null, PatientType.NEW, null),
        EXTRAPUL_RELAPSE("Z0Pplgr23pv", InfectionSite.EXTRAPULMONARY, null, PatientType.PREVIOUSLY_TREATED, PatientType.RELAPSE),
        EXTRAPUL_TREATAFTERFAILURE("jMuqJLXGiwZ", InfectionSite.EXTRAPULMONARY, null, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_FAILURE),
        EXTRAPUL_TREATAFTERLOSSF("NXRLATBzng4", InfectionSite.EXTRAPULMONARY, null, PatientType.PREVIOUSLY_TREATED, PatientType.TREATMENT_AFTER_LOSS_FOLLOW_UP),
        EXTRAPUL_OTHER("alr0Fk49VgI", InfectionSite.EXTRAPULMONARY, null, PatientType.PREVIOUSLY_TREATED, PatientType.OTHER_PREVIOUSLY_TREATED);

        DataElementTB10Block1(String dhis2Id, InfectionSite infectionSite, CaseDefinition caseDefinition, PatientType patientType, PatientType previouslyTreatedType) {
            this.dhis2Id = dhis2Id;
            this.infectionSite = infectionSite;
            this.caseDefinition = caseDefinition;
            this.patientType = patientType;
            this.previouslyTreatedType = previouslyTreatedType;
        }

        /**
         * The data element id in DHIS2
         */
        private String dhis2Id;

        /**
         * The data element infection site
         */
        private InfectionSite infectionSite;

        /**
         * The data element case definition
         */
        private CaseDefinition caseDefinition;

        /**
         * The data element patient type
         */
        private PatientType patientType;

        /**
         * The data element previously treated type
         */
        private PatientType previouslyTreatedType;

        /**
         * Returns the DHIS2 dataElement id that matches with the params.
         *
         * @param infectionSite
         * @param caseDefinition
         * @param patientType
         * @param previouslyTreatedType
         * @return
         */
        public static String findId(InfectionSite infectionSite, CaseDefinition caseDefinition, PatientType patientType, PatientType previouslyTreatedType) {
            // find the id
            if (infectionSite == null || caseDefinition == null || patientType == null) {
                throw new RuntimeException("Infection site, case definition and patient type must not be null");
            }

            for (DataElementTB10Block1 elem : values()) {
                // if it is a pulmonary should test case definition
                if (infectionSite.equals(elem.infectionSite) && infectionSite.equals(InfectionSite.PULMONARY)
                        && caseDefinition.equals(elem.caseDefinition)
                        && patientType.equals(elem.patientType)
                        && ((elem.previouslyTreatedType == null
                        || (patientType.equals(PatientType.PREVIOUSLY_TREATED) && previouslyTreatedType.equals(elem.previouslyTreatedType))))) {
                    return elem.dhis2Id;
                }

                // if it is an extra pulmonary should NOT test case definition, both bacteriologically
                // and clinically are stored in the same data element
                if (infectionSite.equals(elem.infectionSite) && infectionSite.equals(InfectionSite.EXTRAPULMONARY)
                        && patientType.equals(elem.patientType)
                        && ((elem.previouslyTreatedType == null
                        || (patientType.equals(PatientType.PREVIOUSLY_TREATED) && previouslyTreatedType.equals(elem.previouslyTreatedType))))) {
                    return elem.dhis2Id;
                }
            }

            throw new RuntimeException("TB10Block1DataElement: Data element not found");
        }

        public String getDhis2Id() {
            return dhis2Id;
        }
    }

    /**
     * Catalog that holds the ids of DHIS2 categories option combos used by TB 10 Form Block 1
     * COC = Category Option Combo (DHIS2 data delimiter)
     */
    public enum COCAgeGenderBlock1 {

        MALE_UNTIL_4("iTaA28G83g4", Gender.MALE, 0, 4),
        MALE_5_14("JdRjNNX3rAi", Gender.MALE, 5, 14),
        MALE_15_24("YtGJ0vwAYK4", Gender.MALE, 15, 24),
        MALE_25_34("wgNVJSjX2yW", Gender.MALE, 25, 34),
        MALE_35_44("gaFll4BnIPJ", Gender.MALE, 35, 44),
        MALE_45_54("m7qSdvFOBrA", Gender.MALE, 45, 54),
        MALE_55_64("X9eaudnrr7O", Gender.MALE, 55, 64),
        MALE_65_MORE("l9SvaoPCe1i", Gender.MALE, 65, 1000),
        FEMALE_UNTIL_4("WuOOCtJvgb9", Gender.FEMALE, 0, 4),
        FEMALE_5_14("fVhSwKACiNL", Gender.FEMALE, 5, 14),
        FEMALE_15_24("bGcBxjIW4sH", Gender.FEMALE, 15, 24),
        FEMALE_25_34("ediNO2GwnC1", Gender.FEMALE, 25, 34),
        FEMALE_35_44("CeGwyPZeleQ", Gender.FEMALE, 35, 44),
        FEMALE_45_54("s40Yhdzpp6j", Gender.FEMALE, 45, 54),
        FEMALE_55_64("qxd9B4JNW9w", Gender.FEMALE, 55, 64),
        FEMALE_65_MORE("Nry1CjqG3GZ", Gender.FEMALE, 65, 1000);

        /**
         * The category option combo id in DHIS2
         */
        private String dhis2Id;

        /**
         * The category option combo gender
         */
        private Gender gender;

        /**
         * The category option age range initial age
         */
        private Integer iniAge;

        /**
         * The category option combo age range final age
         */
        private Integer endAge;

        COCAgeGenderBlock1(String dhis2Id, Gender gender, Integer iniAge, Integer endAge) {
            this.dhis2Id = dhis2Id;
            this.gender = gender;
            this.iniAge = iniAge;
            this.endAge = endAge;
        }

        /**
         * Returns the DHIS2 categoryOptionCombo id that matches with the age range and gender on params.
         * @param age
         * @param gender
         * @return
         */
        public static String findId(int age, Gender gender){
            if (gender == null) {
                throw new RuntimeException("Gender must not be null");
            }

            for (COCAgeGenderBlock1 cat : values()) {
                if (gender.equals(cat.gender) && age >= cat.iniAge && age <= cat.endAge) {
                    return cat.dhis2Id;
                }
            }

            throw new RuntimeException("CategoryOptionComboAgeGender: Category not found");
        }

        public String getDhis2Id() {
            return dhis2Id;
        }
    }
}
